function [residinfo,parminfo,RNDinfo,testinfo] = LWM_RND_infer(estpar,M1,M2,oprice,strike, rf, TTM,weights,testspec,Disp)
%==========================================================================================
%The function computes the fitted RND estimates and goodness-of-fit Wald
%test based on Li, Nolte and Pham (2023).
%
%INPUT:
%   estpar: 3(M1+M2)-by-1 estimated parameter vector
%   M1: number of Lognormal densities in the mixture
%   M2: number of Weibull densities in the mixture
%   oprice: N-by-1 call option prices
%   strike: N-by-1 strike prices of the options
%   rf: risk-free rate
%   TTM: time to maturity of the options (in years)
%   weights: N-by-1 weight vector. If not provided, assumed equal weights.
%            Default = equal weights
%   testpec: [p,q,alpha] vector of the goodness-of-fit Wald test.
%            Default = [1,1,0.05]
%   Dips: logical, whether Wald test results and plots are displayed.
%         Default = false
%
%OUTPUT:
%   residinfo: a struct containing fitted information 
%              o_fit - fitted call option prices
%              resid - residuals
%              sse - sum squared errors
%              bsiv - observed black implied volatility (based on oprice)
%              bsiv_fit - fitted black implied volatility (based on o_fit)
%   parminfo: a struct containing estimated parameter information
%              estpar_R - reduced/free form of estimated parameter vector (3(M1+M2)-2)-by-1
%              parm_V - var-cov matrix of estimated free parameters
%              parm_se - standard errors of estimated free parameters
%              t_stat - t stats of estimated free parameters
%              p_val  - p values of estimated free parameters
%   RNDinfo: a struct containing fitted RND information
%            PDF - fitted RND
%            RNDi - the mixture-wise fitted PDF
%            RND_bounds - 95% confidence intervals of the fitted RND
%   testinfo: a struct containing the goodness-of-fit Wald test
%            bhat - estimated coefficients of the test auxiliary equation
%            Wstat - Wald statistic
%            pval - pvalue of the Wald stat
%            reg_Y - fitted values of the RHS variable of the auxiliary
%            equation
%==========================================================================================
% This ver: 2023/05/24
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2023). Parametric Risk-Neutral 
%          Density Estimation via Finite Lognormal-Weibull Mixtures
%========================================================================================== 

if nargin < 8 || isempty(weights)
   weights = ones(size(oprice)); 
end

if nargin < 8 || isempty(testspec)
    testspec = [1 1 0.05];
end

if nargin<10
    Disp = false;
end


M = M1+M2;
xval=linspace(strike(1),strike(end),10000)';%a grid to comput the RNDs


N=length(oprice);
%convert to full parameter vector and extract the components
w_vec = estpar(1:M);
f_vec = estpar(M+1:2*M);
sig_vec = estpar(2*M+1:3*M);

Ft = w_vec'*f_vec;

f_vec1 = f_vec(1:M1);
sigma1 = sig_vec(1:M1);
% w_vec1 = w_vec(1:M1);

f_vec2 = f_vec(M1+1:end);
sigma2 = sig_vec(M1+1:end);
% sig_vec2 = w_vec(M1+1:end);


%initialize the structures
residinfo=struct;
RNDinfo=struct;
parminfo=struct;
testinfo=struct;

[~,~,~,o_fit,~, resid]= LWM_MSE(estpar,M1,M2,oprice,strike, rf, TTM,weights,1);
%computing residual statistics
residinfo.o_fit = o_fit;
residinfo.resid = resid;
residinfo.sse  = sum(residinfo.resid.^2);
residinfo.bsiv=blkimpv(Ft,strike,rf,TTM,oprice,[],[],'call');
residinfo.bsiv_fit=blkimpv(Ft,strike,rf,TTM,o_fit,[],[],'call');
%extract RND information
[RND,RNDi]  = LWM_RND_PDF(estpar,M1,M2,xval,TTM);
RNDinfo.PDF = RND;
RNDinfo.RNDi = RNDi;

%Construct confidence bounds for parameters and RND
Wmat = diag(weights);
if M1>0
    [fprice1, delta1, vega1, Gamma1, xi1, eta1] = LN_f(f_vec1', strike,rf,TTM,sigma1', true, true);
else
    fprice1=[];delta1=[];vega1=[];Gamma1=[];xi1=[];eta1=[];
end
if M2>0
    [fprice2, delta2, vega2, Gamma2, xi2, eta2] = WB_f(f_vec2', strike,rf,TTM,sigma2',true,true);
else
    fprice2=[];delta2=[];vega2=[];Gamma2=[];xi2=[];eta2=[];
end
fprices = [fprice1 fprice2];
delta = [delta1 delta2];
vega= [vega1 vega2];
Gamma = [Gamma1 Gamma2];
xi = [xi1 xi2];
eta = [eta1 eta2];

JO=[ fprices(:,1:end-1)-fprices(:,end)-delta(:,end).*(f_vec(1:end-1)'-f_vec(end)) ...
    w_vec(1:end-1)'.*(delta(:,1:end-1)-delta(:,end)) vega.*w_vec' ]; %N-by-(3M-2) jacobian matrix

estpar_R = [w_vec(1:M-1); f_vec(1:M-1); sig_vec];


A_prel = LWM_HessianR(estpar,M1,M2,resid,JO,delta,vega,Gamma,eta,xi,weights)/N;
if min(eig(A_prel))<=0 %if the A estimator is not PSD, use a numerical approximation
    [U,R]=eig(A_prel);
    Reig = diag(R);
    Rpmin = min(Reig(diag(R)>0));
    Reig(Reig< 0) = Rpmin;
    A_prel = U*diag(Reig)*U';
end
% A_prel = JO'*Wmat*JO/N;

B_prel = covnw_f(JO.*(Wmat*residinfo.resid),[],0);
parminfo.estpar_R=estpar_R;
parminfo.parm_V = sinv(A_prel)*B_prel*sinv(A_prel);
parminfo.parm_se= sqrt(diag(parminfo.parm_V));
parminfo.t_stat = estpar_R./parminfo.parm_se;
parminfo.p_val  = 2*normcdf(abs(parminfo.t_stat),'upper');
if M2 >0
    parminfo.parm_out = table(estpar_R,parminfo.parm_se,parminfo.t_stat,parminfo.p_val,  ...
        'VariableNames',{'ParmEst','StdErr','t_stat','p_val'}, ...
        'RowNames',[ 'w_LN'+string(1:M1)  'w_LB'+string(1:M2-1)  'F_LN'+string(1:M1)   ...
        'F_LB'+string(1:M2-1)  'sigma'+string(1:M1) 'k'+string(1:M2) ]');
else
    parminfo.parm_out = table(estpar_R,parminfo.parm_se,parminfo.t_stat,parminfo.p_val,  ...
        'VariableNames',{'ParmEst','StdErr','t_stat','p_val'}, ...
        'RowNames',[ 'w_LN'+string(1:M1-1)   'F_LN'+string(1:M1-1)   ...
        'sigma'+string(1:M1) ]');    
end

%compute the numerical Jacobian of the RND estimates w.r.t. estpar_R
f=@(x) LWM_RND_PDF(parm_convert(x,Ft),M1,M2,xval,TTM);
Jp=gradient_f(f,estpar_R,min(1e-3,min(estpar)*1e-3));
RND_V=Jp*parminfo.parm_V*Jp';
RND_se=sqrt(diag(RND_V));
%construct 95% confidence bounds
RNDinfo.RND_bounds=[RNDinfo.PDF-1.97*RND_se  RNDinfo.PDF+1.97*RND_se];


%construct the Fast Fourier Regression 
p = testspec(1);q=testspec(2);

Kgrid = (strike-Ft)./(strike(end)-strike(1));
C = FFF(Kgrid,p,q);
wresid = sqrt(Wmat)*residinfo.resid;
testinfo.bhat=C\wresid;
Xmat = (C'*C)\C'*(eye(N)*N-sqrt(Wmat)*JO*sinv(A_prel)*JO'*sqrt(Wmat));

% alternative Xmat based on the asymptotic relation under the null:
% lim_{N->infty} JO'*Wmat*JO/N->A_prel.
% Xmat = (C'*C)\C'*(eye(N)-sqrt(Wmat)*JO*sinv(JO'*Wmat*JO)*JO'*sqrt(Wmat))*N;
Vmat = covnw_f(Xmat'.*wresid,[],0);
testinfo.Wstat = N*testinfo.bhat'*sinv(Vmat)*testinfo.bhat;
testinfo.pval = chi2cdf(testinfo.Wstat,p+1+q*2,'upper');
testinfo.reg_Y=C*testinfo.bhat;

if nargout < 1 || Disp == true
    figure; 
    subplot(1,3,1)
    h=plot(xval,[RNDinfo.PDF RNDinfo.RND_bounds]);
    h(1).Color='k';
    h(2).Color='r';h(2).LineStyle='--';
    h(3).Color='r';h(3).LineStyle='--';
    hold on
    plot(xval, RNDi, 'b--', 'LineWidth',0.5);
    ylim([0 max(RNDinfo.PDF*1.5)])
    xline(Ft,'label','F_t')
    legend({'Estimated RND','95% Confidence Bounds'})

    subplot(1,3,2)
    plot(strike,wresid,'ko',strike,testinfo.reg_Y,'r*');
    title('Residuals vs Fourier Regression Fit')   
    legend('WLS residuals','Fourier Regression Fit')    
    disp(parminfo.parm_out)
    
    subplot(1,3,3)
    plot(strike,residinfo.bsiv,'b:',strike,residinfo.bsiv_fit,'r--');
    title('Implied Volatility Fit')   
    legend({'Observed BSIV','Fitted BSIV'})

    
    
    disp('------------------------------------')
    disp('--------Misspecification Test-------')
    disp('------------------------------------')
    disp('    W_stat    p_val')
    disp([testinfo.Wstat testinfo.pval])
end
end

function ret = sinv(X)
%the function computes a numerically stable inverse of a near-singular
%matrix

[m,~]=size(X);
ret = inv(X + 1e-5*eye(m));
end

function p1 = FFF(grid,p,q)
%The function generates the regressors for the fast-Fourier regression
p1=[];
for i=0:p
    p1=[p1 grid.^i];
end
for i=1:q
    p1=[p1 cos(grid.*2.*pi.*i) sin(grid.*2.*pi.*i)];
end
end

function parm = parm_convert(parm_0,Ft)

[T,~]=size(parm_0);
M=ceil(T/3);
w_vec = [parm_0(1:M-1,:); 1 - sum(parm_0(1:M-1,:),1)];
f_vec = [parm_0(M:2*M-2,:); abs((Ft - sum(parm_0(M:2*M-2,:).*w_vec(1:end-1,:),1))./w_vec(end,:))  ];
f_vec(isnan(f_vec))=Ft;
sigma_vec = parm_0(2*M-1:end,:);
parm=[w_vec; f_vec; sigma_vec];

end
